<?php
namespace EssentialBlocks\Core;

use WP_Query;
use EssentialBlocks\Traits\HasSingletone;

/**
 * Load google fonts.
 */

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

class FontLoader {
    use HasSingletone;

    /**
     * Google fonts to enqueue
     *
     * @access public
     * @var array
     */
    public static $gfonts  = [];
    private static $blocks = [];

    /**
     * The Constructor.
     */
    public function __construct() {
        add_filter( 'the_content', [$this, 'the_content'], 1 );
    }

    public function get_fonts( $blocks ) {
        if ( empty( $blocks ) ) {
            return self::$gfonts;
        }
        $loaded_fonts = [];
        foreach ( $blocks as $block ) {
            $loaded_fonts = array_merge( $loaded_fonts, self::load_gfonts( $block['attrs'] ) );
        }

        return array_merge( self::$gfonts, $loaded_fonts );
    }

    public function get_blocks_from_template_parts() {
        $template_part_query = new WP_Query(
            [
                'post_type'      => 'wp_template_part',
                'post_status'    => 'publish',
                'posts_per_page' => -1,
                'no_found_rows'  => true
            ]
        );

        if ( $template_part_query->have_posts() ) {
            $blocks = [];
            while ( $template_part_query->have_posts() ): $template_part_query->the_post();
                $blocks = $this->get_blocks( get_the_content(), $blocks );
            endwhile;

            self::$blocks = array_merge( self::$blocks, $blocks );
        }

        wp_reset_postdata();
        wp_reset_query();
    }

    public function get_blocks_from_templates() {
        $template_part_query = new WP_Query(
            [
                'post_type'      => 'wp_template',
                'post_status'    => 'publish',
                'posts_per_page' => -1,
                'no_found_rows'  => true
            ]
        );

        if ( $template_part_query->have_posts() ) {
            $blocks = [];
            while ( $template_part_query->have_posts() ): $template_part_query->the_post();
                $blocks = $this->get_blocks( get_the_content(), $blocks );
            endwhile;

            self::$blocks = array_merge( self::$blocks, $blocks );
        }

        wp_reset_postdata();
        wp_reset_query();
    }

    public function get_blocks( $content, $blocks = [] ) {
        $_blocks = [];
        if ( is_array( $content ) ) {
            $_blocks = $content;
        }

        if ( is_string( $content ) && has_blocks( $content ) ) {
            $_blocks = parse_blocks( $content );
        }

        if ( isset( $_blocks ) && ! is_array( $_blocks ) ) {
            return [];
        }

        foreach ( $_blocks as $key => $block ) {
            if ( $block['blockName'] === null ) {
                continue;
            }
            // for reusable blocks
            if ( isset( $block['attrs']['ref'] ) ) {
                $_content = get_post( $block['attrs']['ref'] );
                if ( ! empty( $_content->post_content ) ) {
                    $_blocks = $this->get_blocks( $_content->post_content, $_blocks );
                }
            }

            // for inner blocks
            if ( ! empty( $block['innerBlocks'] ) ) {
                $_blocks = $this->get_blocks( $block['innerBlocks'], $_blocks );
            }
        }

        /* TODO: need check again why it is given null */
        $_blocks = $_blocks === null ? [] : $_blocks;
        $blocks  = $blocks === null ? [] : $blocks;

        return array_merge( $_blocks, $blocks );
    }

    public function the_content( $content ) {
        // get blocks from template parts
        $this->get_blocks_from_template_parts();
        $this->get_blocks_from_templates();

        // get blocks from current content
        self::$blocks = array_merge( self::$blocks, $this->get_blocks( $content, self::$blocks ) );
        self::$gfonts = array_merge( self::$gfonts, $this->get_fonts( self::$blocks ) );

        $this->fonts_loader();

        return $content;
    }

    public static function get_fonts_family( $attributes ) {
        $keys             = preg_grep( '/^(\w+)FontFamily/i', array_keys( $attributes ), 0 );
        $googleFontFamily = [];
        foreach ( $keys as $key ) {
            $googleFontFamily[$attributes[$key]] = $attributes[$key];
        }
        return $googleFontFamily;
    }

    /**
     * Load fonts.
     *
     * @access public
     */
    public function fonts_loader( $handle_name = 'eb-block-fonts' ) {
        $eb_settings = get_option( 'eb_settings', [] );
        $googleFont  = ! empty( $eb_settings['googleFont'] ) ? $eb_settings['googleFont'] : 'true';

        if ( 'false' !== $googleFont ) {
            $fonts = self::$gfonts;

            if (  ( $key = array_search( 'Default', $fonts ) ) !== false ) {
                unset( $fonts[$key] );
            }
            if ( ! empty( $fonts ) ) {
                $gfonts      = '';
                $gfonts_attr = ':100,100italic,200,200italic,300,300italic,400,400italic,500,500italic,600,600italic,700,700italic,800,800italic,900,900italic';
                foreach ( $fonts as $font ) {
                    $gfonts .= str_replace( ' ', '+', trim( $font ) ) . $gfonts_attr . '|';
                }
                if ( ! empty( $gfonts ) ) {
                    $query_args = [
                        'family' => $gfonts
                    ];
                    wp_register_style(
                        $handle_name,
                        add_query_arg( $query_args, '//fonts.googleapis.com/css' ),
                        []
                    );
                    wp_enqueue_style( $handle_name );
                }
                // Reset.
                $gfonts = '';
            }
        }
    }
    /**
     * save google font family for blocks
     * @since 4.0.0
     * @param array $fonts get fonts family array
     *
     * @return array
     */
    public static function load_gfonts( $attributes ) {
        $fonts        = self::get_fonts_family( $attributes );
        self::$gfonts = array_unique( array_merge( self::$gfonts, $fonts ) );

        return self::$gfonts;
    }
}
